/***************************************************************************
 *   Copyright (C) 2015 by Laboratoire d'Economie Forestière               *
 *   http://ffsm-project.org                                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version, given the compliance with the     *
 *   exceptions listed in the file COPYING that is distribued together     *
 *   with this file.                                                       *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
//#include <QtGui>   // Qt4
#include <QtWidgets> // Qt5

#include "MainWindow.h"
#include "ScenarioSelectionWidget.h"
#include "QApplication"

using namespace std;


// *************** Initializzation functions...   *******************

/** 

It setup the Gui from the QTDesiger autogenerated code and connect various GUI signal/slots

*/
MainWindow::MainWindow() {
  yearSBLabel=NULL;
  mainSBLabel=NULL;
  for (uint i=0;i<MaxRecentFiles;i++) recentFileActions[i] = NULL;
  separatorAction=NULL;

  setupUi(this);
  unsavedStatus=false;
  curModelFileName="data/ffsmInput.ods";
  curBaseDirectory = QApplication::applicationDirPath();
  curBaseDirectory.append("/data/");
  //curBaseDirectory = "data/";
  outputDirName="output/";
  setCurrentLogFileName("");
  createStatusBar();
  curLogFileName ="";
  debugMsgsEnable = true;

  for (int i = 0; i < MaxRecentFiles; ++i) {
    recentFileActions[i] = new QAction(this);
    recentFileActions[i]->setVisible(false);
    connect(recentFileActions[i], SIGNAL(triggered()), this, SLOT(openRecentFile()));
  }

  separatorAction = menuFile->addSeparator();
  for (int i = 0; i < MaxRecentFiles; ++i)
    menuFile->addAction(recentFileActions[i]);
  menuFile->addSeparator();
  menuFile->addAction(actionExit);

  readSettings();
  modelMainThread.setInputFileName(curModelFileName);
  //modelMainThread.setBaseDirectory(curBaseDirectory);

  // Status viewer....
  statusView->setColumnCount(2);
  statusView->setHeaderLabels(QStringList()<< tr ("Label") << tr ("Value"));
  statusView->clear();
  statusView->sortByColumn(0);
  statusView->setFocus(); //????




  /*
  DONE: statusView should be implemented like this:

  Model
    -> year
    -> total plots
    -> rented plots
    -> abandoned plots
  Managers
    -> Manager_farmer
      -> number of agents
  Agents
    Agent_0
      -> Type
      -> ID
      -> mould
      -> owned plots
      ...
    Agent_1
      -> Type
      -> ID
      -> mould
      -> owned plots
      ...
    ...
  */

  qRegisterMetaType<string>("string"); // allows string objects to be thread-safely queued within signal-slots comunications
  qRegisterMetaType<QString>("QString");
  qRegisterMetaType< QVector<QString> >("QVector<QString>");


  connect(actionRun, SIGNAL(triggered()), this, SLOT(startModelMainThread()));
  connect(actionPause, SIGNAL(triggered()), this, SLOT(pauseOrResumeModelMainThread()));
  connect(actionStop, SIGNAL(triggered()), this, SLOT(stopModelMainThread()));
  connect(actionExit, SIGNAL(triggered()), this, SLOT(close()));
  connect(actionSaveLog, SIGNAL(triggered()), this, SLOT(save()));
  connect(actionSaveLogAs, SIGNAL(triggered()), this, SLOT(saveAs()));
  connect(actionLoadConfiguration, SIGNAL(triggered()), this, SLOT(open()));
  connect(actionHideDebugMsgs, SIGNAL(triggered(bool)), this, SLOT(hideDebugMsgs(bool)));
  connect(actionAboutRegMAS, SIGNAL(triggered()), this, SLOT(about()));
  connect(actionRegMASDocumentation, SIGNAL(triggered()), this, SLOT(showDocumentation()));
  connect(actionFitMap, SIGNAL(triggered()), mapBox, SLOT(fitInWindow()));
  connect(this, SIGNAL(resized()),mapBox, SLOT(fitInWindow()));
  connect(viewResultsButton, SIGNAL(clicked()),this, SLOT(openResults()));

  connect(&modelMainThread, SIGNAL(upgradeLogArea(const QString&)), this, SLOT(processLogArea(const QString&)));
  connect(&modelMainThread, SIGNAL(addLayerToGui(QString, QString)), this, SLOT( addLayer(QString, QString)));  
  connect(layerSelector, SIGNAL(activated(int)), this, SLOT(switchToLayerFromLayerSelector(int)));
  connect(&modelMainThread, SIGNAL(updatePixelToGui(QString, int, int, QColor)), this, SLOT (updatePixel(QString, int, int, QColor)));
  connect(&modelMainThread, SIGNAL(updateImageToGui(QString, QImage)), this, SLOT (updateImage(QString, QImage)));
  connect(&modelMainThread, SIGNAL(setOutputDirNameToGui(string)), this, SLOT(setOutputDirName(string)));
  connect(&modelMainThread, SIGNAL(setGUIUnsavedStatus(bool)), this, SLOT(setUnsavedStatus(bool)));
  connect(&modelMainThread, SIGNAL(sendScenarioOptionsToGUI(const QVector<QString> &)), this, SLOT( receiveScenarioOptions(const QVector<QString> &)   ));

  // Scenario selection widget...
  scenarioWidget = new ScenarioSelectionWidget(this);
  connect(scenarioWidget->scenarioSelector, SIGNAL( activated(const QString&)), scenarioWidget, SLOT( close()));
  connect(scenarioWidget->scenarioSelector, SIGNAL( activated(const QString&)), &modelMainThread, SLOT( retrieveScenarioNameFromGUI(const QString &)));
  //connect(scenarioWidget, SIGNAL( selectedScenarioName(const QString&)), scenarioWidget, SLOT( close()));
  //connect(scenarioWidget, SIGNAL( selectedScenarioName(const QString&)), &modelMainThread, SLOT( retrieveScenarioNameFromGUI(const QString &)));

  // Model tree viewer...
  connect(&modelMainThread, SIGNAL( treeViewerItemChangeValueToGui(string, string)  ), this, SLOT( treeViewerItemChangeValue(string, string) ));
  connect(&modelMainThread, SIGNAL( treeViewerItemRemoveToGui(string)  ), this, SLOT( treeViewerItemRemove(string) ));
  connect(&modelMainThread, SIGNAL( treeViewerAddItemToGui(string, string, string)  ), this, SLOT( treeViewerAddItem(string, string, string) ));
  connect(&modelMainThread, SIGNAL( fitInWindowToGui()), mapBox, SLOT(fitInWindow()));

  connect(mapBox, SIGNAL(  queryRequestOnPx(int, int, bool) ), &modelMainThread, SLOT ( checkQuery(int, int, bool) ) );
  connect(&modelMainThread,SIGNAL(publishQueryResults(const QString&)), pxInfoArea, SLOT (setHtml(const QString&)));
  connect(&modelMainThread,SIGNAL(activateTab(int)), tabWidget, SLOT (setCurrentIndex(int)));

  connect(&modelMainThread, SIGNAL( resetGUIForNewSimulation()  ), this, SLOT( resetGUIForNewSimulation() ));

}

void
MainWindow::createStatusBar() {
  yearSBLabel = new QLabel(" 2000 ");
  yearSBLabel->setAlignment(Qt::AlignHCenter);
  yearSBLabel->setMinimumSize(yearSBLabel->sizeHint());

  mainSBLabel = new QLabel;
  mainSBLabel->setIndent(3);

  statusBar()->addWidget(yearSBLabel);
  statusBar()->addWidget(mainSBLabel, 1);

  yearSBLabel->setText("0");
  mainSBLabel->setText("Welcome to FFSM!");

  connect(&modelMainThread, SIGNAL(upgradeYearSBLabelToGui(const QString&)), yearSBLabel, SLOT(setText(const QString&)));
  connect(&modelMainThread, SIGNAL(upgradeMainSBLabelToGui(const QString&)), mainSBLabel, SLOT(setText(const QString&)));

}

// Manage the event of closing the application
void
MainWindow::closeEvent(QCloseEvent *event) {
  if (okToContinue()) {
    writeSettings();
    modelMainThread.stop();
    modelMainThread.wait();
    event->accept();
  } else {
    event->ignore();
    }
} 

void
MainWindow::resizeEvent (QResizeEvent *event) {
  emit resized();
}


// ***************  open model / log saving functions..  ************************

void
MainWindow::setCurrentLogFileName(const QString &fileName) {
  curLogFileName = fileName;
}

void
MainWindow::setCurrentModelFileName(const QString &fileName) {
  curModelFileName = fileName;
  //setWindowModified(false);
  modelMainThread.setInputFileName(curModelFileName);
  
  QString shownName = "Untitled";
  if (!curModelFileName.isEmpty()) {
    shownName = strippedName(curModelFileName);
    recentFiles.removeAll(curModelFileName);
    recentFiles.prepend(curModelFileName);
    updateRecentFileActions();
  }
  setWindowTitle(tr("%2 - [%1]").arg(shownName).arg(tr("FFSM - Forest Sector Simulator")));
}

QString
MainWindow::strippedName(const QString &fullFileName) {
  return QFileInfo(fullFileName).fileName();
}

void
MainWindow::updateRecentFileActions() {
  QMutableStringListIterator i(recentFiles);
  while (i.hasNext()) {
    if (!QFile::exists(i.next()))
      i.remove();
  }

  for (int j = 0; j < MaxRecentFiles; ++j) {
    if (j < recentFiles.count()) {
      QString text = tr("&%1 %2")
            .arg(j + 1)
            .arg(strippedName(recentFiles.at(j)));
      //cerr <<text.toStdString()<<endl;
      recentFileActions[j]->setText(text);
      recentFileActions[j]->setData(recentFiles.at(j));
      recentFileActions[j]->setVisible(true);
    } else {
      recentFileActions[j]->setVisible(false);
    }
  }
  separatorAction->setVisible(!recentFiles.isEmpty());
}

bool
MainWindow::okToContinue() {
  if (modelMainThread.isRunning()) {
    int t = QMessageBox::warning(
      this,                                       // parent
      tr("FFSM"),                               // title
      tr("The model is still running.\n"    // message
        "Do you want to stop it?"),
      QMessageBox::Yes | QMessageBox::Default,    // 1st button
      QMessageBox::Cancel | QMessageBox::Escape   // 3rd button
    );
    if (t == QMessageBox::Yes) {
      modelMainThread.stop();
      modelMainThread.wait();
    } else if (t == QMessageBox::Cancel) {
      return false;
    }
  }

  if (unsavedStatus) {
    int r = QMessageBox::warning(
      this,                                       // parent
      tr("FFSM"),                               // title
      tr("The model log has not been saved.\n"    // message
        "Do you want to save it?"),
      QMessageBox::Yes ,                          // 1st button
      QMessageBox::No | QMessageBox::Default,     // 2nd button
      QMessageBox::Cancel | QMessageBox::Escape   // 3rd button
    );
    if (r == QMessageBox::Yes) {
      return save();
    } else if (r == QMessageBox::Cancel) {
      return false;
    }
  }
  return true;
}

void
MainWindow::open() {
  if (okToContinue()) {
    QString fileName = QFileDialog::getOpenFileName(
      this,
      tr("Load model file.."),
      "data/",
      tr("OpenDocument Spreadsheet (*.ods)\n" "All files (*.*)")
    );
    if (!fileName.isEmpty()){
      statusBar()->showMessage(tr("Loaded new FFSM model file"), 2000);
      setCurrentModelFileName(fileName);
      // getting the baseData path information...
      QFileInfo info(fileName);
      QString path;
      path = info.absolutePath();
      path = path+"/";
      curBaseDirectory = path;
      //modelMainThread.setBaseDirectory(curBaseDirectory);
    }
    }
}

void
MainWindow::readSettings() {
  QSettings settings("LEF", "FFSM");
  recentFiles = settings.value("recentFiles").toStringList();
  updateRecentFileActions();
}

void
MainWindow::openRecentFile() {
  if (okToContinue()) {
    QAction *action = qobject_cast<QAction *>(sender());
    if (action){
      curModelFileName=action->data().toString();
      setCurrentModelFileName(curModelFileName);
      // getting the baseData path information...
      QFileInfo info(curModelFileName);
      QString path;
      path = info.absolutePath();
      path = path+"/";
      curBaseDirectory = path;
      //modelMainThread.setBaseDirectory(curBaseDirectory);
    }
  }
}

bool
MainWindow::save() {
  if (curLogFileName.isEmpty()) {
    return saveAs();
  } else {
    cerr <<(curLogFileName.toStdString())<<endl;
    cerr <<(outputDirName.toStdString())<<endl;
    return saveLogFile(curLogFileName);
  }
  unsavedStatus = false;
  return true;
}

bool
MainWindow::saveAs() {
  QString logFileName = QFileDialog::getSaveFileName(
    this,
    tr("Save output log"),
    outputDirName,
    tr("Log files (*.log)\n" "All files (*.*)")
  );
  if (logFileName.isEmpty())
    return false;
  return saveLogFile(logFileName);
  unsavedStatus = false;
  return true;
}

bool
MainWindow::saveLogFile(const QString &logFileName) {
  QFile file(logFileName);
  if (!file.open(QIODevice::WriteOnly)) {
  QMessageBox::warning(this, tr("FFSM"),
     tr("Cannot write log file file %1:\n%2.")
    .arg(file.fileName())
    .arg(file.errorString()));
    return false;
  }
  //QString logAreaContent = logArea->toHtml();
  QString logAreaContent = logArea->toPlainText(); // Also available "toHtml()"
  QTextStream stream( &file );
  stream << logAreaContent;
  file.close();

  setCurrentLogFileName(logFileName);
  statusBar()->showMessage(tr("Log file saved"), 2000);
  unsavedStatus = false;
  return true;
}

void MainWindow::writeSettings() {
  QSettings settings("LEF", "FFSM");
    settings.setValue("recentFiles", recentFiles);
}

// ****************** Main thread controllers ***********************

void
MainWindow::startModelMainThread() {
  if (modelMainThread.isRunning()) {
    return ;
    cout <<"It seems that the model is already running..."<<endl;
    } else {  
    logArea->clear();
    modelMainThread.start();
    unsavedStatus=true;
    }
}

void
MainWindow::stopModelMainThread() {
  if (! modelMainThread.isRunning()) {
    return ;
  } else {
    modelMainThread.stop();
    modelMainThread.wait();
  }
}

void
MainWindow::pauseOrResumeModelMainThread() {
  modelMainThread.pauseOrResume();
}

// ************ display px info **********************
/*void
MainWindow::sendQueryToMainThread(int px_ID){
  modelMainThread.pause();
  //modelMainThread.wait();
  modelMainThread.computeQuery(px_ID);
  modelMainThread.resume();
}*/


// ************** Map operations ***********************


/**
Perform all the operation needed when adding a new layer:
 - add a layer to mapBox;
 - add the layer to layerSelector;
 - (NOTNEEDED: add the layer to layerLegend); Not needed any longer, as legend was dropped in name of the Model Status Viewer
*/
void
MainWindow::addLayer(QString layerName_h, QString layerLabel_h) {
  static int counter =0;
  mapBox->addLayer(layerName_h);
  layerSelector->addItem(layerLabel_h,layerName_h);
  // first layer added only. it is not needed as MapBox::addLayer() and QComboBox automatically switch to the new value if it is the first one :-))
  //if (counter == 0) switchToLayer(layerName_h); 
  update();
  counter ++;
}

/**
Perform all the operation needed when switching layer:
 - call mapBox to switch its current layer;
 - call layerLegend to switch its layer);
I don't think it is used anywhere, but any how.. it is here...
*/
void
MainWindow::switchToLayer(QString layerName_h) {
  mapBox->switchToLayer(layerName_h);
  int index = mapBox->getLayerIndex(layerName_h);
  layerSelector->setCurrentIndex(index);
  update();
}

void
MainWindow::switchToLayerFromLayerSelector(int layerIndex_h) {
  QString layerName= layerSelector->itemData(layerIndex_h, Qt::UserRole ).toString();
  mapBox->switchToLayer(layerName);
  update();
} 

void
MainWindow::updatePixel(QString layerName_h, int x_h, int y_h, QColor color_h) {
  mapBox->updatePixel(layerName_h,x_h,y_h,color_h.rgb());
  update();
}

void
MainWindow::updateImage(QString layerName_h, const QImage &image_h) {
  mapBox->updateImage(layerName_h, image_h);
  update();
}

// ************** Status viewer operations ***********************
void
MainWindow::treeViewerItemChangeValue(string itemID, string newValue) {

  map<string, QTreeWidgetItem*>::iterator p;
  p=svIndex.find(itemID);
  if(p != svIndex.end())
    p->second->setText(1,newValue.c_str());
  else {
    QString tempString;
    QString tempString2 = itemID.c_str();
    tempString = "**** ERROR, Coud not change value for item "+tempString2+" in the Model Status Viewer. Item doesn't found.";
    logArea->append(tempString);
  }
  return;
  
}

void
MainWindow::treeViewerItemRemove(string itemID) {
  map<string, QTreeWidgetItem*>::iterator p;
  p=svIndex.find(itemID);
  if(p != svIndex.end()){
    QTreeWidgetItem *parent = p->second->parent();
    int index;
    if (parent) {
      index = parent->indexOfChild(p->second); //DONE: check if it works !!! While it should not ???? After 15 years of simulation agents should be deleted, but htey are still here in the tree.. mayme it is true it is NOT working!!! To be checken. 20071108: It works, it works.. agents are deleted when go out of the model
      delete parent->takeChild(index);
      svIndex.erase(p);
    } else {
      QString tempString = "**** ERROR, I will not delete a top level item in the Model Satus Viewer";
      logArea->append(tempString);
    }

  }
  else {
    QString tempString;
    QString tempString2 = itemID.c_str();
    tempString = "**** ERROR, Coud not delete for item "+tempString2+" in the Model Status Viewer. Item doesn't found.";
    //logArea->append(tempString); //20080111 lots of this errors when re-starting a simulation, so hidding them
  }
  return;
}

void
MainWindow::treeViewerAddItem(string text, string itemID, string parentID){
  // searching for the parent item...
  map<string, QTreeWidgetItem*>::iterator p;
  QTreeWidgetItem *parentItem;

  p=svIndex.find(parentID);
  if(p != svIndex.end()){
    parentItem = p->second;
    QTreeWidgetItem *node = new QTreeWidgetItem(parentItem);
    svIndex.insert(pair<string, QTreeWidgetItem*>(itemID, node));
    node->setText(0, text.c_str());
  }
  else {
    QString tempString;
    QString tempString2 = itemID.c_str();
    QString tempString3 = parentID.c_str();
    tempString = "**** ERROR, Coud not add sub item "+tempString2+" to the Model Status Viewer. Parent item ("+tempString3+") doesn't found.";
    logArea->append(tempString);
  }

}

// ************** Other ***********************
void
MainWindow::processLogArea(const QString& message_h){
  if(debugMsgsEnable){
    logArea->append(message_h);
  }
  else {
    if( ! message_h.startsWith("*DEBUG")){
      logArea->append(message_h);
    }
  }
}

void
MainWindow::hideDebugMsgs(bool hide){
  if(hide) debugMsgsEnable = false;
  else debugMsgsEnable = true;
}

void
MainWindow::about(){
  QMessageBox::about(this, tr("About FFSM"),
    tr("<h2>FFSM</h2>"
       "<p>Copyright &copy; 2012 Laboratoire d'Economie Forestière - LEF"
       "<br/>"
       "<p>FFSM is a flexible, spatially explicit, coupled resource and economic simulator of the Forest Sector, "
       "designed for long-term simulations of effects of government policies "
       "over different forest systems."
       "<br>It is released under the GNU GPL licence."
       "<p>For documentation and credits please refer to the project site:"
       "<br><a href=\"http://www.ffsm-project.org\">http://www.ffsm-project.org</a>"
    ));
}

void
MainWindow::showDocumentation(){
  QMessageBox::question(this, tr("FFSM Documentation"), // QMessageBox::information or QMessageBox::question
    tr("<h2>FFSM Documentation</h2>"
       "<p align=\"justify\">FFSM documentation is organised in three main categories: "
       "<p align=\"left\">(1) <b>official documentation</b> "
       "(comprising the <i>User Manual</i> and the <i>Reference Manual</i>); <br>(2) <b>contributed "
       "documentation</b> (<i>wiki</i>);<br>(3) <b>community project</b> (<i>forum</i> and <i>mailing list</i>). "
       "<p align=\"justify\">The documentation is located at "
       "<a href=\"http://www.ffsm-project.org/doc\">http://www.ffsm-project.org/doc</a>"
       "<p align=\"justify\">If you have chosen to instal a local copy of the documentation, "
       "you can access it also from the <i>Start menu</i>-><i>Programs</i>-><i>FFSM</i> "
       "(MS Windows) or directly from the following links (Linux):"
       "<br><a href=\"doc/userManual/regmasUserManual.pdf\">User Manual</a> "
       "&nbsp;-&nbsp;<a href=\"doc/referenceManual/html/index.html\">Reference Manual</a> "
       "<p>Tips:"
       "<br> - right click on a pixel to get its value across the layers;"
       "<br> - use the mouse and its wheel over the map to zoom/scroll it;"
       "</p>"
    ));
}

void
MainWindow::resetGUIForNewSimulation(){

  static int simulationCounter = 0;
  //reset map <string, QTreeWidgetItem*>       svIndex and clean the tree widget
  statusView->clear();
  map<string, QTreeWidgetItem*>::iterator p;
  //for(p=svIndex.begin(); p= svIndex.end(); p++){
    //delete p->second; // no need because they are destroyed already from statusView->clear();
  //}
  svIndex.clear();

  QTreeWidgetItem* svGeneralNode = new QTreeWidgetItem(statusView);
  svIndex.insert(pair<string, QTreeWidgetItem*>("general", svGeneralNode));
  svGeneralNode -> setText(0, "General");
  QTreeWidgetItem* svYearItem = new QTreeWidgetItem(svGeneralNode);
  svIndex.insert(pair<string, QTreeWidgetItem*>("general_year", svYearItem));
  svYearItem->setText(0, "year");
  svYearItem->setText(1, "0");
  QTreeWidgetItem* svTotalPlotsItem = new QTreeWidgetItem(svGeneralNode);
  svIndex.insert(pair<string, QTreeWidgetItem*>("general_total plots", svTotalPlotsItem));
  svTotalPlotsItem->setText(0, "total plots");
  svTotalPlotsItem->setText(1, "0");
  QTreeWidgetItem* svTotalLandItem = new QTreeWidgetItem(svGeneralNode);
  svIndex.insert(pair<string, QTreeWidgetItem*>("general_total land", svTotalLandItem));
  svTotalLandItem->setText(0, "total land");
  QTreeWidgetItem* svTotalAgrLandItem = new QTreeWidgetItem(svGeneralNode);
  svIndex.insert(pair<string, QTreeWidgetItem*>("general_total agr land", svTotalAgrLandItem));
  svTotalAgrLandItem->setText(0, "total agr land");
  QTreeWidgetItem* svOwnedAgrLandItem = new QTreeWidgetItem(svGeneralNode);
  svIndex.insert(pair<string, QTreeWidgetItem*>("general_owned agr land", svOwnedAgrLandItem));
  svOwnedAgrLandItem->setText(0, "owned agr land");
  QTreeWidgetItem* svRentedAgrLandItem = new QTreeWidgetItem(svGeneralNode);
  svIndex.insert(pair<string, QTreeWidgetItem*>("general_rented agr land", svRentedAgrLandItem));
  svRentedAgrLandItem->setText(0, "rented agr land");

  QTreeWidgetItem* svManagersNode = new QTreeWidgetItem(statusView);
  svIndex.insert(pair<string, QTreeWidgetItem*>("managers", svManagersNode));
  svManagersNode->setText(0,"Managers");

  QTreeWidgetItem* svAgentsNode = new QTreeWidgetItem(statusView);
  svIndex.insert(pair<string, QTreeWidgetItem*>("agents", svAgentsNode));
  svAgentsNode->setText(0,"Agents");

  // reset layer selector
  layerSelector->clear();
  // reset pixel info area
  pxInfoArea->setHtml("<i>Tip: Right click over a plot to retrieve its values across layers.</i>");
  // reset log area
  logArea->clear();
  // reset map

  if (simulationCounter) logArea->append("***WARNING: You are running more simulations from the GUI without closing/reopening it. It should works, but there are no guarantees. The best way is to run only one simulation from the GUI, eventually closing and opening FFSM again for further simulations.");
  simulationCounter++;
  
}

void
MainWindow::receiveScenarioOptions(const QVector<QString> &scenarios_h){

  //for(uint i=0;i<scenarios_h.size();i++){
  //  cout << scenarios_h.at(i).toStdString() << endl;
  //} // stange.. it works like expected !!!!

  scenarioWidget->receiveScenarioOptions(scenarios_h);
  scenarioWidget->show();
  scenarioWidget->scenarioSelector->setFocus();
  //scenarioWidget->scenarioSelector->grabMouse();
  //scenarioWidget->scenarioSelector->grabKeyboard();


}

void
MainWindow::openResults() {
  //QLabel *label = new QLabel("Hello World!");
  //label->show();
  //string aaa = curBaseDirectory.toStdString();
  //cout << "curBaseDirectory " << aaa << endl;
  //cout << "outputDirName: " << outputDirName.toStdString() << endl;
  QUrl resultsUrl(curBaseDirectory+outputDirName+"results/results.ods", QUrl::TolerantMode);
  QDesktopServices::openUrl(resultsUrl);

}
